// ═══════════════════════════════════════════════════════════════════════
// Loopbook — Unified Extension
// Manager view (Ann/Jack) + Advisor view (Tom)
// Role is determined by the ## Team section in TASKS.md
// ═══════════════════════════════════════════════════════════════════════

// ── State ────────────────────────────────────────────────────────────
let fileHandle = null;
let currentUser = null;    // { name: 'Tom', role: 'advisor' }
let teamMembers = [];      // [{ name, role }] from ## Team
let taskData = [];
let somedayData = [];
let doneData = [];

// Manager state
let currentFilter = 'all';
let expandedIdx = null;
let editingIdx = null;

// Advisor state
let focusIdx = 0;

const $ = (id) => document.getElementById(id);
const OWNERS_FALLBACK = ['Jack', 'Tom', 'Ann'];

// ── Boot ─────────────────────────────────────────────────────────────
document.addEventListener('DOMContentLoaded', function() {
  $('btnConnect').addEventListener('click', connectFile);
  $('refreshBtn').addEventListener('click', refreshData);
  $('userChip').addEventListener('click', switchUser);

  // Manager controls
  $('addTaskBtn').addEventListener('click', addNewTask);
  $('somedayHead').addEventListener('click', function() { togCollapse('somedayBody', 'somedayCaret'); });
  $('doneHead').addEventListener('click', function() { togCollapse('doneBody', 'doneCaret'); });
  $('importToggle').addEventListener('click', function() { toggleImport(); });
  $('importBtn').addEventListener('click', doImport);
  $('importCancelBtn').addEventListener('click', function() {
    $('importText').value = '';
    $('importHint').textContent = '';
    toggleImport(false);
  });
  $('importText').addEventListener('input', updateImportHint);
  $('importText').addEventListener('keydown', function(e) {
    if (e.key === 'Enter' && (e.ctrlKey || e.metaKey)) { e.preventDefault(); doImport(); }
  });

  // Advisor controls
  $('doneBtn').addEventListener('click', markDone);
  $('skipBtn').addEventListener('click', skipTask);
  $('undoBtn').addEventListener('click', undoLastDone);
  $('queueToggle').addEventListener('click', toggleQueue);
  $('advisorDoneToggle').addEventListener('click', function() {
    $('advisorDoneBody').classList.toggle('open');
    $('advisorDoneArrow').classList.toggle('open');
  });

  // Escape key
  document.addEventListener('keydown', function(e) {
    if (e.key === 'Escape') {
      if (editingIdx !== null) { editingIdx = null; renderManager(); }
      else if (expandedIdx !== null) { expandedIdx = null; renderManager(); }
    }
  });

  // Check for saved user
  chrome.storage.local.get(['loopbookUser'], function(result) {
    if (result.loopbookUser) {
      currentUser = result.loopbookUser;
      // Still need file handle — show setup but pre-selected
    }
  });
});

// ── File connection ──────────────────────────────────────────────────
async function connectFile() {
  try {
    [fileHandle] = await window.showOpenFilePicker({
      types: [{ description: 'Markdown', accept: { 'text/markdown': ['.md'] } }],
      multiple: false
    });

    // Parse file to get team members
    const f = await fileHandle.getFile();
    const text = await f.text();
    parseFullFile(text);

    // If we have a saved user and they exist in the team, skip picker
    if (currentUser && teamMembers.find(function(m) { return m.name === currentUser.name; })) {
      enterApp();
      return;
    }

    // Show name picker
    if (teamMembers.length > 0) {
      showNamePicker();
    } else {
      // No Team section — fall back to defaults
      teamMembers = [
        { name: 'Tom', role: 'advisor' },
        { name: 'Ann', role: 'manager' },
        { name: 'Jack', role: 'manager' }
      ];
      showNamePicker();
    }
  } catch (e) {}
}

function showNamePicker() {
  $('btnConnect').classList.add('ok');
  $('btnConnect').textContent = '✓ Connected';
  $('namePicker').classList.remove('hidden');

  const list = $('nameList');
  list.innerHTML = '';

  for (var i = 0; i < teamMembers.length; i++) {
    var m = teamMembers[i];
    var btn = document.createElement('button');
    btn.className = 'name-btn';

    var dot = document.createElement('span');
    dot.className = 'name-dot';
    dot.style.background = getDotColor(m.name);
    btn.appendChild(dot);

    btn.appendChild(document.createTextNode(m.name));

    var role = document.createElement('span');
    role.className = 'name-role';
    role.textContent = m.role;
    btn.appendChild(role);

    btn.setAttribute('data-name', m.name);
    btn.setAttribute('data-role', m.role);
    btn.addEventListener('click', function() {
      var name = this.getAttribute('data-name');
      var role = this.getAttribute('data-role');
      currentUser = { name: name, role: role };
      chrome.storage.local.set({ loopbookUser: currentUser });
      enterApp();
    });

    list.appendChild(btn);
  }
}

function enterApp() {
  $('setup').classList.add('hidden');
  $('appHeader').classList.remove('hidden');

  // Set user chip
  $('userName').textContent = currentUser.name;
  $('userDot').style.background = getDotColor(currentUser.name);

  if (currentUser.role === 'advisor') {
    $('viewLabel').textContent = '';
    $('advisorView').classList.remove('hidden');
    $('managerView').classList.add('hidden');
    $('importBar').classList.add('hidden');
  } else {
    $('viewLabel').textContent = 'Manager';
    $('managerView').classList.remove('hidden');
    $('importBar').classList.remove('hidden');
    $('advisorView').classList.add('hidden');
    buildFilters();
  }

  renderAll();
}

function switchUser() {
  // Reset back to setup
  $('appHeader').classList.add('hidden');
  $('managerView').classList.add('hidden');
  $('advisorView').classList.add('hidden');
  $('importBar').classList.add('hidden');

  currentUser = null;
  chrome.storage.local.remove('loopbookUser');

  if (fileHandle && teamMembers.length > 0) {
    $('setup').classList.remove('hidden');
    showNamePicker();
  } else {
    $('setup').classList.remove('hidden');
    $('btnConnect').classList.remove('ok');
    $('btnConnect').textContent = 'Open TASKS.md';
    $('namePicker').classList.add('hidden');
  }
}

async function refreshData() {
  if (!fileHandle) return;
  try {
    var f = await fileHandle.getFile();
    parseFullFile(await f.text());
    renderAll();
    showToast('Refreshed', 'success');
  } catch (e) {
    showToast('Error loading file', 'error');
  }
}

// ── Parse TASKS.md ───────────────────────────────────────────────────
function parseFullFile(md) {
  var secs = { 'Team': [], 'Tasks': [], 'Someday': [], 'Done': [] };
  var cur = null;

  for (var li = 0; li < md.split('\n').length; li++) {
    var line = md.split('\n')[li];
    var sm = line.match(/^## (.+)/);
    if (sm) { cur = sm[1].trim(); continue; }
    if (!cur) continue;

    // Team section
    if (cur === 'Team') {
      var tm = line.match(/^- (.+?)\s*\|\s*(advisor|manager)\s*$/i);
      if (tm) {
        secs['Team'].push({ name: tm[1].trim(), role: tm[2].toLowerCase() });
      }
      continue;
    }

    // Task lines
    var taskM = line.match(/^- \[([ x])\] \*\*(.+?)\*\*(.*)$/);
    if (taskM) {
      var done = taskM[1] === 'x', name = taskM[2], rest = taskM[3];
      var owner = ''; var om = rest.match(/👤\s*(\w+)/); if (om) owner = om[1];
      var due = ''; var dm = rest.match(/due\s+(.+?)(?:\s*—|$)/); if (dm) due = dm[1].trim();
      var og = rest.match(/ongoing\s*\((.+?)\)/); if (og) due = og[1];
      var contact = ''; var cm = rest.match(/📋\s*(.+?)(?:\s*—|$)/); if (cm) contact = cm[1].trim();
      var category = ''; var catm = rest.match(/🏷️\s*(.+?)(?:\s*—|$)/); if (catm) category = catm[1].trim();
      if (secs[cur]) secs[cur].push({ name: name, owner: owner, due: due, done: done, desc: '', subtasks: [], contact: contact, category: category });
      continue;
    }
    var sub = line.match(/^\s+- (.+)$/);
    if (sub && cur && secs[cur] && secs[cur].length) {
      secs[cur][secs[cur].length - 1].subtasks.push(sub[1]);
    }
  }

  // Convert subtasks to desc
  for (var secName in secs) {
    if (secName === 'Team') continue;
    var sec = secs[secName];
    for (var si = 0; si < sec.length; si++) {
      if (sec[si].subtasks && sec[si].subtasks.length) {
        sec[si].desc = sec[si].subtasks.join('\n');
      }
    }
  }

  teamMembers = secs['Team'].length > 0 ? secs['Team'] : [];
  taskData = secs['Tasks'] || [];
  somedayData = secs['Someday'] || [];
  doneData = secs['Done'] || [];
}

// ── Render router ────────────────────────────────────────────────────
function renderAll() {
  if (!currentUser) return;
  if (currentUser.role === 'advisor') {
    renderAdvisor();
  } else {
    renderManager();
  }
}

// ═══════════════════════════════════════════════════════════════════════
// MANAGER VIEW
// ═══════════════════════════════════════════════════════════════════════

function getOwners() {
  if (teamMembers.length > 0) return teamMembers.map(function(m) { return m.name; });
  return OWNERS_FALLBACK;
}

function buildFilters() {
  var filters = $('filters');
  filters.innerHTML = '';
  var allNames = getOwners();

  var filterData = [{ key: 'all', label: 'All', dot: '' }];
  for (var fi = 0; fi < allNames.length; fi++) {
    filterData.push({ key: allNames[fi], label: allNames[fi], dot: 'fdot-custom' });
  }

  for (var fdi = 0; fdi < filterData.length; fdi++) {
    var fd = filterData[fdi];
    var btn = document.createElement('button');
    btn.className = 'ftab' + (fd.key === 'all' ? ' active' : '');
    if (fd.key !== 'all') {
      var dot = document.createElement('span');
      dot.className = 'fdot';
      dot.style.background = getDotColor(fd.key);
      btn.appendChild(dot);
    }
    btn.appendChild(document.createTextNode(fd.label));
    btn.setAttribute('data-key', fd.key);
    btn.addEventListener('click', function() {
      currentFilter = this.getAttribute('data-key');
      document.querySelectorAll('.ftab').forEach(function(t) { t.classList.remove('active'); });
      this.classList.add('active');
      expandedIdx = null; editingIdx = null;
      renderManager();
    });
    filters.appendChild(btn);
  }
}

function renderManager() {
  renderTasks();
  renderSomeday();
  renderDone();
}

function renderTasks() {
  var el = $('taskList');
  var firstVisible = -1;
  for (var i = 0; i < taskData.length; i++) {
    if (currentFilter === 'all' || taskData[i].owner.toLowerCase() === currentFilter.toLowerCase()) {
      firstVisible = i; break;
    }
  }

  el.innerHTML = '';
  for (var i = 0; i < taskData.length; i++) {
    var t = taskData[i];
    var show = currentFilter === 'all' || t.owner.toLowerCase() === currentFilter.toLowerCase()
      || (!t.owner && currentFilter === 'all');
    if (!show) continue;

    var isFirst = i === firstVisible;
    var isExpanded = expandedIdx === i;
    var isEditing = editingIdx === i;

    var card = document.createElement('div');
    card.className = 'task' + (isFirst ? ' current' : '');
    card.draggable = true;
    card.setAttribute('data-idx', i);

    var row = document.createElement('div');
    row.className = 'task-row';

    var dot = document.createElement('div');
    dot.className = 'tdot';
    dot.style.background = getDotColor(t.owner);

    var nameEl = document.createElement('div');
    nameEl.className = 'tname';
    nameEl.textContent = t.name;

    var dueSp = document.createElement('span');
    if (t.due) {
      dueSp.className = 'tdue' + (isDueP(t.due) ? ' overdue' : '');
      dueSp.textContent = formatDue(t.due);
    }

    var chev = document.createElement('span');
    chev.className = 'tchev' + (isExpanded ? ' open' : '');
    chev.textContent = '▶';

    row.appendChild(dot);
    row.appendChild(nameEl);
    if (t.due) row.appendChild(dueSp);
    row.appendChild(chev);
    card.appendChild(row);

    // Click expand
    (function(idx) {
      row.addEventListener('click', function() {
        if (editingIdx !== null && editingIdx !== idx) return;
        expandedIdx = expandedIdx === idx ? null : idx;
        editingIdx = null;
        renderManager();
      });
    })(i);

    // Drag events
    card.addEventListener('dragstart', function(e) { e.dataTransfer.effectAllowed = 'move'; this.classList.add('dragging'); });
    card.addEventListener('dragover', function(e) { e.preventDefault(); e.dataTransfer.dropEffect = 'move'; });
    card.addEventListener('dragenter', function(e) { e.preventDefault(); this.classList.add('drag-over'); });
    card.addEventListener('dragleave', function() { this.classList.remove('drag-over'); });
    card.addEventListener('dragend', function() { this.classList.remove('dragging'); document.querySelectorAll('.drag-over').forEach(function(el) { el.classList.remove('drag-over'); }); });
    (function(toIdx) {
      card.addEventListener('drop', function(e) {
        e.preventDefault();
        this.classList.remove('drag-over');
        var fromEl = document.querySelector('.task.dragging');
        if (!fromEl) return;
        var fromIdx = parseInt(fromEl.getAttribute('data-idx'));
        if (fromIdx === toIdx) return;
        var item = taskData.splice(fromIdx, 1)[0];
        taskData.splice(toIdx, 0, item);
        expandedIdx = null; editingIdx = null;
        renderManager(); saveFile();
      });
    })(i);

    // Expanded detail
    if (isExpanded) {
      var detail = document.createElement('div');
      detail.className = 'task-detail open';

      if (isEditing) {
        renderEditMode(detail, t, i);
      } else {
        renderViewMode(detail, t, i);
      }
      card.appendChild(detail);
    }

    el.appendChild(card);
  }
}

function renderEditMode(detail, t, idx) {
  var owners = getOwners();

  // Subject
  var subRow = document.createElement('div'); subRow.className = 'td-row';
  var subLabel = document.createElement('div'); subLabel.className = 'td-label'; subLabel.textContent = 'Subject';
  var subInput = document.createElement('input'); subInput.className = 'td-input'; subInput.type = 'text'; subInput.value = t.name; subInput.id = 'edit-name-' + idx;
  subRow.appendChild(subLabel); subRow.appendChild(subInput); detail.appendChild(subRow);

  // Description
  var descRow = document.createElement('div'); descRow.className = 'td-row';
  var descLabel = document.createElement('div'); descLabel.className = 'td-label'; descLabel.textContent = 'Description';
  var descInput = document.createElement('textarea'); descInput.className = 'td-textarea'; descInput.value = t.desc || ''; descInput.placeholder = 'Add details...'; descInput.id = 'edit-desc-' + idx;
  descRow.appendChild(descLabel); descRow.appendChild(descInput); detail.appendChild(descRow);

  // Owner + Due
  var metaRow = document.createElement('div'); metaRow.className = 'td-meta';

  var ownerWrap = document.createElement('div'); ownerWrap.className = 'td-row'; ownerWrap.style.flex = '1';
  var ownerLabel = document.createElement('div'); ownerLabel.className = 'td-label'; ownerLabel.textContent = 'Owner';
  var ownerSelect = document.createElement('select'); ownerSelect.className = 'td-select'; ownerSelect.id = 'edit-owner-' + idx;
  var emptyOpt = document.createElement('option'); emptyOpt.value = ''; emptyOpt.textContent = 'Select...'; ownerSelect.appendChild(emptyOpt);
  for (var oi = 0; oi < owners.length; oi++) {
    var opt = document.createElement('option'); opt.value = owners[oi]; opt.textContent = owners[oi];
    if (t.owner === owners[oi]) opt.selected = true;
    ownerSelect.appendChild(opt);
  }
  ownerWrap.appendChild(ownerLabel); ownerWrap.appendChild(ownerSelect); metaRow.appendChild(ownerWrap);

  var dueWrap = document.createElement('div'); dueWrap.className = 'td-row'; dueWrap.style.flex = '1';
  var dueLabel = document.createElement('div'); dueLabel.className = 'td-label'; dueLabel.textContent = 'Due date';
  var dueInput = document.createElement('input'); dueInput.className = 'td-date'; dueInput.type = 'date'; dueInput.id = 'edit-due-' + idx;
  if (t.due) { try { var d = new Date(t.due); if (!isNaN(d)) dueInput.value = d.toISOString().split('T')[0]; } catch (e) {} }
  dueWrap.appendChild(dueLabel); dueWrap.appendChild(dueInput); metaRow.appendChild(dueWrap);
  detail.appendChild(metaRow);

  // Buttons
  var btnBar = document.createElement('div'); btnBar.className = 'td-edit-bar';
  var delBtn = document.createElement('button'); delBtn.className = 'ebtn ebtn-del'; delBtn.textContent = 'Delete';
  (function(i) { delBtn.addEventListener('click', function() {
    taskData.splice(i, 1); expandedIdx = null; editingIdx = null; renderManager(); saveFile(); showToast('Deleted', 'success');
  }); })(idx);
  var cancelBtn = document.createElement('button'); cancelBtn.className = 'ebtn ebtn-cancel'; cancelBtn.textContent = 'Cancel';
  cancelBtn.addEventListener('click', function() { editingIdx = null; renderManager(); });
  var saveBtn = document.createElement('button'); saveBtn.className = 'ebtn ebtn-save'; saveBtn.textContent = 'Save';
  (function(i, task) { saveBtn.addEventListener('click', function() {
    task.name = document.getElementById('edit-name-' + i).value.trim() || task.name;
    task.desc = document.getElementById('edit-desc-' + i).value.trim();
    task.owner = document.getElementById('edit-owner-' + i).value;
    var dv = document.getElementById('edit-due-' + i).value;
    if (dv) { var dp = new Date(dv + 'T00:00:00'); task.due = dp.toLocaleDateString('en-US', { weekday: 'short', month: 'short', day: 'numeric', year: 'numeric' }); } else { task.due = ''; }
    task.subtasks = task.desc ? task.desc.split('\n').filter(function(l) { return l.trim(); }) : [];
    editingIdx = null; renderManager(); saveFile(); showToast('Saved ✓', 'success');
  }); })(idx, t);
  btnBar.appendChild(delBtn); btnBar.appendChild(cancelBtn); btnBar.appendChild(saveBtn);
  detail.appendChild(btnBar);
}

function renderViewMode(detail, t, idx) {
  // Description
  var descRow = document.createElement('div'); descRow.className = 'td-row';
  var descLabel = document.createElement('div'); descLabel.className = 'td-label'; descLabel.textContent = 'Description';
  var descText = document.createElement('div'); descText.className = 'td-desc' + (t.desc ? '' : ' empty');
  descText.textContent = t.desc || 'No description';
  descRow.appendChild(descLabel); descRow.appendChild(descText); detail.appendChild(descRow);

  // Meta badges
  var meta = document.createElement('div'); meta.className = 'td-meta';
  if (t.owner) {
    var ob = document.createElement('span'); ob.className = 'td-owner-badge';
    var od = document.createElement('span'); od.className = 'tdot'; od.style.background = getDotColor(t.owner);
    od.style.width = '5px'; od.style.height = '5px';
    ob.appendChild(od); ob.appendChild(document.createTextNode(' ' + t.owner)); meta.appendChild(ob);
  }
  if (t.contact) {
    var cb = document.createElement('span'); cb.className = 'td-owner-badge'; cb.textContent = '👤 ' + t.contact; meta.appendChild(cb);
  }
  if (t.due) {
    var db = document.createElement('span'); db.className = 'td-due-badge' + (isDueP(t.due) ? ' overdue' : '');
    db.textContent = formatDue(t.due); meta.appendChild(db);
  }
  if (t.category) {
    var catBadge = document.createElement('span'); catBadge.className = 'td-owner-badge'; catBadge.style.fontSize = '10px';
    catBadge.textContent = t.category; meta.appendChild(catBadge);
  }
  detail.appendChild(meta);

  // Buttons
  var btnBar = document.createElement('div'); btnBar.className = 'td-edit-bar';
  var markDoneBtn = document.createElement('button'); markDoneBtn.className = 'ebtn ebtn-del';
  markDoneBtn.textContent = 'Mark done'; markDoneBtn.style.color = '#3D7A3D'; markDoneBtn.style.borderColor = '#A8C9A8';
  (function(i) { markDoneBtn.addEventListener('click', function() {
    var item = taskData.splice(i, 1)[0]; item.done = true; doneData.unshift(item);
    expandedIdx = null; renderManager(); saveFile(); showToast('Done ✓', 'success');
  }); })(idx);
  var editBtn = document.createElement('button'); editBtn.className = 'ebtn ebtn-edit'; editBtn.textContent = 'Edit';
  (function(i) { editBtn.addEventListener('click', function() { editingIdx = i; renderManager(); }); })(idx);
  btnBar.appendChild(markDoneBtn); btnBar.appendChild(editBtn);
  detail.appendChild(btnBar);
}

function renderSomeday() {
  var sv = somedayData.filter(function(t) {
    return currentFilter === 'all' || t.owner.toLowerCase() === currentFilter.toLowerCase() || !t.owner;
  });
  $('somedayN').textContent = sv.length;

  var body = $('somedayBody');
  body.innerHTML = '';
  for (var i = 0; i < somedayData.length; i++) {
    var t = somedayData[i];
    var show = currentFilter === 'all' || t.owner.toLowerCase() === currentFilter.toLowerCase() || !t.owner;
    if (!show) continue;

    var row = document.createElement('div'); row.className = 'stask';
    var dot = document.createElement('div'); dot.className = 'tdot'; dot.style.background = getDotColor(t.owner);
    var nameEl = document.createElement('div'); nameEl.className = 'tname'; nameEl.textContent = t.name;

    var promoteBtn = document.createElement('button'); promoteBtn.className = 'promote-btn'; promoteBtn.textContent = '↑ Tasks';
    (function(idx) { promoteBtn.addEventListener('click', function() {
      var item = somedayData.splice(idx, 1)[0]; taskData.push(item); renderManager(); saveFile(); showToast('Moved to Tasks', 'success');
    }); })(i);

    var dismissBtn = document.createElement('button'); dismissBtn.className = 'dismiss-btn'; dismissBtn.textContent = '×';
    (function(idx) { dismissBtn.addEventListener('click', function() {
      somedayData.splice(idx, 1); renderManager(); saveFile();
    }); })(i);

    row.appendChild(dot); row.appendChild(nameEl); row.appendChild(promoteBtn); row.appendChild(dismissBtn);
    body.appendChild(row);
  }
}

function renderDone() {
  var dv = doneData.filter(function(t) {
    return currentFilter === 'all' || t.owner.toLowerCase() === currentFilter.toLowerCase() || !t.owner;
  });
  $('doneN').textContent = dv.length;

  var body = $('doneBody');
  body.innerHTML = '';
  for (var di = 0; di < doneData.length; di++) {
    var t = doneData[di];
    var show = currentFilter === 'all' || t.owner.toLowerCase() === currentFilter.toLowerCase() || !t.owner;
    if (!show) continue;
    var row = document.createElement('div'); row.className = 'stask done';
    var dot = document.createElement('div'); dot.className = 'tdot'; dot.style.background = getDotColor(t.owner);
    var nameEl = document.createElement('div'); nameEl.className = 'tname'; nameEl.textContent = t.name;

    var undoBtn = document.createElement('button'); undoBtn.className = 'promote-btn'; undoBtn.textContent = '↩ Undo';
    (function(idx) { undoBtn.addEventListener('click', function() {
      var item = doneData.splice(idx, 1)[0]; item.done = false; taskData.unshift(item);
      renderManager(); saveFile();
    }); })(di);

    row.appendChild(dot); row.appendChild(nameEl); row.appendChild(undoBtn); body.appendChild(row);
  }
}

function addNewTask() {
  taskData.push({ name: 'New task', owner: '', due: '', done: false, desc: '', subtasks: [], contact: '', category: '' });
  expandedIdx = taskData.length - 1;
  editingIdx = taskData.length - 1;
  renderManager();
  var nameInput = document.getElementById('edit-name-' + (taskData.length - 1));
  if (nameInput) { nameInput.select(); nameInput.focus(); }
}

function togCollapse(bid, cid) {
  document.getElementById(bid).classList.toggle('open');
  document.getElementById(cid).classList.toggle('open');
}

// ═══════════════════════════════════════════════════════════════════════
// ADVISOR VIEW
// ═══════════════════════════════════════════════════════════════════════

function renderAdvisor() {
  // Filter to advisor's tasks
  var myName = currentUser.name;
  var myTasks = taskData.filter(function(t) {
    return !t.owner || t.owner === myName;
  });

  renderFocus(myTasks);
  renderQueue(myTasks);
  renderAdvisorDone();
}

function renderFocus(myTasks) {
  var area = $('focusArea');
  area.innerHTML = '';

  // Show/hide undo button based on whether there are done items for this user
  var myName = currentUser.name;
  var hasDone = doneData.some(function(t) { return !t.owner || t.owner === myName; });
  if (hasDone) { $('undoBtn').classList.remove('hidden'); } else { $('undoBtn').classList.add('hidden'); }

  if (focusIdx >= myTasks.length || myTasks.length === 0) {
    $('doneBar').classList.add('hidden');
    area.innerHTML =
      '<div class="empty-state">' +
        '<div class="empty-icon">✓</div>' +
        '<div class="empty-title">You\'re all clear</div>' +
        '<div class="empty-sub">No tasks right now.<br>Hit ↻ to check for updates.</div>' +
      '</div>';
    // Still show done/undo drawers even when no active tasks
    return;
  }

  $('doneBar').classList.remove('hidden');
  var t = myTasks[focusIdx];

  var label = document.createElement('div'); label.className = 'up-next-label';
  label.textContent = focusIdx === 0 ? 'Focus' : 'Up next';
  area.appendChild(label);

  var card = document.createElement('div'); card.className = 'focus-card';

  var subj = document.createElement('div'); subj.className = 'focus-subject'; subj.textContent = t.name;
  card.appendChild(subj);

  if (t.contact) {
    var ct = document.createElement('div'); ct.className = 'focus-contact'; ct.textContent = '👤 ' + t.contact;
    card.appendChild(ct);
  }

  if (t.desc) {
    var div = document.createElement('div'); div.className = 'focus-divider'; card.appendChild(div);
    // Figure out who the manager is (the person who preps notes)
    var managerName = 'the team';
    for (var mi = 0; mi < teamMembers.length; mi++) {
      if (teamMembers[mi].role === 'manager') { managerName = teamMembers[mi].name; break; }
    }
    var nlabel = document.createElement('div'); nlabel.className = 'focus-notes-label';
    nlabel.textContent = 'Notes from ' + managerName;
    card.appendChild(nlabel);

    var notes = document.createElement('div'); notes.className = 'focus-notes'; notes.textContent = t.desc;
    card.appendChild(notes);
  }

  if (t.category || t.due) {
    if (t.category) {
      var cat = document.createElement('span'); cat.className = 'focus-category'; cat.textContent = t.category;
      card.appendChild(cat);
    }
    if (t.due) {
      var dueEl = document.createElement('span');
      dueEl.className = 'focus-due ' + getDueClass(t.due);
      dueEl.textContent = formatDueAdvisor(t.due);
      card.appendChild(dueEl);
    }
  }

  area.appendChild(card);
}

function renderQueue(myTasks) {
  var body = $('queueBody');
  body.innerHTML = '';

  var upcoming = myTasks.slice(focusIdx + 1);
  $('queueCount').textContent = upcoming.length > 0 ? '(' + upcoming.length + ')' : '';

  if (upcoming.length === 0) {
    $('queueToggle').classList.add('hidden');
    return;
  }

  $('queueToggle').classList.remove('hidden');

  for (var qi = 0; qi < upcoming.length; qi++) {
    var t = upcoming[qi];
    var row = document.createElement('div'); row.className = 'queue-item';
    var num = document.createElement('div'); num.className = 'queue-num'; num.textContent = (qi + 2);
    var nameEl = document.createElement('div'); nameEl.className = 'queue-name'; nameEl.textContent = t.name;
    row.appendChild(num); row.appendChild(nameEl); body.appendChild(row);
  }
}

function renderAdvisorDone() {
  var body = $('advisorDoneBody');
  body.innerHTML = '';

  var myName = currentUser.name;
  var myDone = [];
  for (var i = 0; i < doneData.length; i++) {
    if (!doneData[i].owner || doneData[i].owner === myName) {
      myDone.push({ item: doneData[i], realIdx: i });
    }
  }

  $('advisorDoneCount').textContent = myDone.length > 0 ? '(' + myDone.length + ')' : '';

  if (myDone.length === 0) {
    $('advisorDoneToggle').classList.add('hidden');
    return;
  }

  $('advisorDoneToggle').classList.remove('hidden');

  for (var di = 0; di < myDone.length; di++) {
    var t = myDone[di].item;
    var row = document.createElement('div');
    row.className = 'queue-item';

    var nameEl = document.createElement('div');
    nameEl.className = 'queue-name';
    nameEl.style.textDecoration = 'line-through';
    nameEl.style.opacity = '0.6';
    nameEl.textContent = t.name;

    var undoBtn = document.createElement('button');
    undoBtn.className = 'undo-inline-btn';
    undoBtn.textContent = 'Undo';
    (function(realIdx) {
      undoBtn.addEventListener('click', function() {
        undoAdvisorDoneItem(realIdx);
      });
    })(myDone[di].realIdx);

    row.appendChild(nameEl);
    row.appendChild(undoBtn);
    body.appendChild(row);
  }
}

async function markDone() {
  var myName = currentUser.name;
  var myTasks = taskData.filter(function(t) { return !t.owner || t.owner === myName; });
  if (focusIdx >= myTasks.length) return;

  var item = myTasks[focusIdx];
  item.done = true;

  // Remove from taskData
  var realIdx = taskData.indexOf(item);
  if (realIdx > -1) taskData.splice(realIdx, 1);
  doneData.unshift(item);

  // Recompute myTasks
  myTasks = taskData.filter(function(t) { return !t.owner || t.owner === myName; });
  if (focusIdx >= myTasks.length) focusIdx = Math.max(0, myTasks.length - 1);

  renderAdvisor();
  await saveFile();
}

function undoLastDone() {
  var myName = currentUser.name;
  // Find the most recent done item that belongs to this advisor
  for (var i = 0; i < doneData.length; i++) {
    if (!doneData[i].owner || doneData[i].owner === myName) {
      var item = doneData.splice(i, 1)[0];
      item.done = false;
      taskData.unshift(item);
      focusIdx = 0;
      renderAdvisor();
      saveFile();
      return;
    }
  }
}

function undoAdvisorDoneItem(doneIdx) {
  var item = doneData.splice(doneIdx, 1)[0];
  item.done = false;
  taskData.unshift(item);
  focusIdx = 0;
  renderAdvisor();
  saveFile();
}

function skipTask() {
  var myName = currentUser.name;
  var myTasks = taskData.filter(function(t) { return !t.owner || t.owner === myName; });
  if (myTasks.length <= 1) return;

  var item = myTasks[focusIdx];
  // Move in taskData to end
  var realIdx = taskData.indexOf(item);
  if (realIdx > -1) {
    taskData.splice(realIdx, 1);
    taskData.push(item);
  }

  myTasks = taskData.filter(function(t) { return !t.owner || t.owner === myName; });
  if (focusIdx >= myTasks.length) focusIdx = 0;

  renderAdvisor();
  saveFile();
}

function toggleQueue() {
  $('queueBody').classList.toggle('open');
  $('queueArrow').classList.toggle('open');
}

// ═══════════════════════════════════════════════════════════════════════
// REDTAIL IMPORT (Manager only)
// ═══════════════════════════════════════════════════════════════════════

var RT_CATEGORIES = ['General Information', 'Family Planning', 'Customer Service', 'Financial Planning', 'Insurance', 'Investment'];

function parseRedtail(text) {
  var lines = text.split('\n').filter(function(l) { return l.trim(); });
  var tasks = [];
  for (var i = 0; i < lines.length; i++) {
    var line = lines[i].trim();
    if (line.match(/^When\s*Subject/i) || line.match(/^When\t/i)) continue;
    if (line.match(/^Showing \d+ to \d+/i)) continue;
    if (line.match(/^Previous|^Next|^›|^‹/i)) continue;
    var tabs = line.split('\t');
    if (tabs.length >= 4) { var parsed = parseRedtailTabs(tabs); if (parsed) { tasks.push(parsed); continue; } }
    var parsed = parseRedtailRegex(line);
    if (parsed) tasks.push(parsed);
  }
  return tasks;
}

function parseRedtailTabs(cols) {
  var when = (cols[0] || '').trim();
  var subject = (cols[1] || '').trim();
  var type = (cols[2] || '').trim();
  var assignedTo = (cols[3] || '').trim();
  var contact = (cols[4] || '').trim();
  var category = (cols[6] || cols[5] || '').trim();
  if (!subject) return null;
  return { name: subject, owner: mapRedtailOwner(assignedTo), due: parseRedtailDate(when), done: false, desc: '', subtasks: [], contact: contact, category: category, type: type };
}

function parseRedtailRegex(line) {
  var dateMatch = line.match(/^((?:Jan|Feb|Mar|Apr|May|Jun|Jul|Aug|Sep|Oct|Nov|Dec)\s+\d{1,2}(?:,?\s+\d{4})?\s+\d{1,2}:\d{2}\s*[AP]M)/i);
  if (!dateMatch) return null;
  var when = dateMatch[1];
  var rest = line.substring(dateMatch[0].length);
  var bestTypeIdx = -1, bestType = '';
  var typeRe = /(Task|Appointment)(?=\d+\s+users?|[A-Z][a-z]+\s+[A-Z][a-z]+)/gi;
  var m;
  while ((m = typeRe.exec(rest)) !== null) { bestTypeIdx = m.index; bestType = m[1]; }
  if (bestTypeIdx === -1) {
    var simpleRe = /(Task|Appointment)/gi;
    while ((m = simpleRe.exec(rest)) !== null) { bestTypeIdx = m.index; bestType = m[1]; }
  }
  if (bestTypeIdx === -1) return null;
  var subject = rest.substring(0, bestTypeIdx).trim();
  rest = rest.substring(bestTypeIdx + bestType.length);
  var assignedTo = '';
  var assignMatch = rest.match(/^(\d+\s+users?|[A-Z][a-z]+\s+[A-Z][a-z]+)/);
  if (assignMatch) { assignedTo = assignMatch[1]; rest = rest.substring(assignMatch[0].length); }
  var category = '';
  for (var ci = 0; ci < RT_CATEGORIES.length; ci++) {
    if (rest.endsWith(RT_CATEGORIES[ci])) { category = RT_CATEGORIES[ci]; rest = rest.substring(0, rest.length - category.length).trim(); break; }
  }
  var contact = rest.trim();
  if (!subject) return null;
  return { name: subject, owner: mapRedtailOwner(assignedTo), due: parseRedtailDate(when), done: false, desc: '', subtasks: [], contact: contact, category: category, type: bestType };
}

function mapRedtailOwner(assignedTo) {
  var s = (assignedTo || '').toLowerCase();
  // Match against team members
  for (var i = 0; i < teamMembers.length; i++) {
    if (s.includes(teamMembers[i].name.toLowerCase())) return teamMembers[i].name;
  }
  // Fallback hardcoded
  if (s.includes('jack')) return 'Jack';
  if (s.includes('tom')) return 'Tom';
  if (s.includes('ann')) return 'Ann';
  return '';
}

function parseRedtailDate(when) {
  if (!when) return '';
  var cleaned = when.trim();
  var withYear = cleaned.match(/^((?:Jan|Feb|Mar|Apr|May|Jun|Jul|Aug|Sep|Oct|Nov|Dec)\s+\d{1,2},?\s+\d{4})/i);
  var noYear = cleaned.match(/^((?:Jan|Feb|Mar|Apr|May|Jun|Jul|Aug|Sep|Oct|Nov|Dec)\s+\d{1,2})/i);
  var dateStr;
  if (withYear) { dateStr = withYear[1]; }
  else if (noYear) { dateStr = noYear[1] + ', 2026'; }
  else { return ''; }
  try { var d = new Date(dateStr); if (isNaN(d)) return ''; return d.toLocaleDateString('en-US', { weekday: 'short', month: 'short', day: 'numeric', year: 'numeric' }); } catch (e) { return ''; }
}

function sortByDue(tasks) {
  return tasks.sort(function(a, b) {
    var da = a.due ? new Date(a.due) : null;
    var db = b.due ? new Date(b.due) : null;
    var now = new Date();
    if (da && !db) return -1;
    if (!da && db) return 1;
    if (!da && !db) return 0;
    var aOver = da < now, bOver = db < now;
    if (aOver && !bOver) return -1;
    if (!aOver && bOver) return 1;
    return da - db;
  });
}

function doImport() {
  var text = $('importText').value.trim();
  if (!text) { showToast('Paste your Redtail tasks first', 'error'); return; }
  var parsed = parseRedtail(text);
  if (parsed.length === 0) { showToast('No tasks found — check the paste format', 'error'); return; }
  for (var pi = 0; pi < parsed.length; pi++) { taskData.push(parsed[pi]); }
  taskData = sortByDue(taskData);
  $('importText').value = '';
  $('importHint').textContent = '';
  toggleImport(false);
  renderManager();
  saveFile();
  showToast(parsed.length + ' tasks imported', 'success');
}

function toggleImport(forceOpen) {
  var body = $('importBody');
  var arrow = $('importArrow');
  var isOpen = body.style.display !== 'none';
  var shouldOpen = forceOpen !== undefined ? forceOpen : !isOpen;
  body.style.display = shouldOpen ? 'block' : 'none';
  arrow.classList.toggle('open', shouldOpen);
  if (shouldOpen) $('importText').focus();
}

function updateImportHint() {
  var text = $('importText').value.trim();
  if (!text) { $('importHint').textContent = ''; return; }
  var lines = text.split('\n').filter(function(l) {
    var t = l.trim();
    return t && !t.match(/^When\s*Subject/i) && !t.match(/^When\t/i) && !t.match(/^Showing \d+/i);
  });
  $('importHint').textContent = lines.length + ' rows detected';
}

// ═══════════════════════════════════════════════════════════════════════
// SAVE
// ═══════════════════════════════════════════════════════════════════════

async function saveFile() {
  if (!fileHandle) return;
  try {
    var perm = await fileHandle.queryPermission({ mode: 'readwrite' });
    if (perm !== 'granted') await fileHandle.requestPermission({ mode: 'readwrite' });

    var md = '# Loopbook\n\n';

    // Write Team section
    if (teamMembers.length > 0) {
      md += '## Team\n';
      for (var ti = 0; ti < teamMembers.length; ti++) {
        md += '- ' + teamMembers[ti].name + ' | ' + teamMembers[ti].role + '\n';
      }
      md += '\n';
    }

    md += '## Tasks\n';
    for (var i = 0; i < taskData.length; i++) {
      var t = taskData[i];
      var line = '- [' + (t.done ? 'x' : ' ') + '] **' + t.name + '**';
      if (t.owner) line += ' — 👤 ' + t.owner;
      if (t.due) line += ' — due ' + t.due;
      if (t.contact) line += ' — 📋 ' + t.contact;
      if (t.category) line += ' — 🏷️ ' + t.category;
      md += line + '\n';
      if (t.desc) {
        var descLines = t.desc.split('\n');
        for (var dl = 0; dl < descLines.length; dl++) {
          if (descLines[dl].trim()) md += '  - ' + descLines[dl].trim() + '\n';
        }
      }
    }
    md += '\n## Someday\n';
    for (var si = 0; si < somedayData.length; si++) {
      var line = '- [ ] **' + somedayData[si].name + '**';
      if (somedayData[si].owner) line += ' — 👤 ' + somedayData[si].owner;
      md += line + '\n';
    }
    md += '\n## Done\n';
    for (var di = 0; di < doneData.length; di++) {
      var line = '- [x] **' + doneData[di].name + '**';
      if (doneData[di].owner) line += ' — 👤 ' + doneData[di].owner;
      md += line + '\n';
    }

    var w = await fileHandle.createWritable();
    await w.write(md);
    await w.close();
  } catch (e) {
    showToast('Error saving', 'error');
  }
}

// ═══════════════════════════════════════════════════════════════════════
// HELPERS
// ═══════════════════════════════════════════════════════════════════════

function getDotColor(name) {
  var n = (name || '').toLowerCase();
  if (n === 'tom') return '#5B8CB8';
  if (n === 'ann') return '#8B7EC8';
  if (n === 'jack') return '#3D6B5E';
  return '#D9D3C7';
}

function isDueP(s) {
  try { var d = new Date(s); if (isNaN(d)) return false; var today = new Date(); today.setHours(0,0,0,0); return d < today; } catch (e) { return false; }
}

function formatDue(s) {
  if (!s) return '';
  try {
    var d = new Date(s); if (isNaN(d)) return s;
    var now = new Date(); now.setHours(0,0,0,0);
    var diff = Math.round((d - now) / (1000*60*60*24));
    if (diff === 0) return 'Today';
    if (diff === 1) return 'Tomorrow';
    if (diff === -1) return 'Yesterday';
    if (diff < -1) return Math.abs(diff) + 'd overdue';
    if (diff <= 7) return d.toLocaleDateString('en-US', { weekday: 'short' });
    return d.toLocaleDateString('en-US', { month: 'short', day: 'numeric' });
  } catch (e) {} return s;
}

function formatDueAdvisor(s) {
  if (!s) return '';
  try {
    var d = new Date(s); if (isNaN(d)) return s;
    var now = new Date(); now.setHours(0,0,0,0);
    var diff = Math.round((d - now) / (1000*60*60*24));
    if (diff === 0) return 'Due today';
    if (diff === 1) return 'Due tomorrow';
    if (diff === -1) return '1 day overdue';
    if (diff < -1) return Math.abs(diff) + ' days overdue';
    if (diff <= 7) return 'Due ' + d.toLocaleDateString('en-US', { weekday: 'long' });
    return 'Due ' + d.toLocaleDateString('en-US', { month: 'short', day: 'numeric' });
  } catch (e) {} return s;
}

function getDueClass(s) {
  if (!s) return 'normal';
  try { var d = new Date(s); if (isNaN(d)) return 'normal'; var now = new Date(); now.setHours(0,0,0,0); var diff = Math.round((d - now) / (1000*60*60*24)); if (diff < 0) return 'overdue'; if (diff <= 2) return 'upcoming'; return 'normal'; } catch (e) { return 'normal'; }
}

var toastTimer;
function showToast(msg, type) {
  var t = $('toast');
  t.textContent = msg;
  t.className = 'toast show ' + (type || '');
  clearTimeout(toastTimer);
  toastTimer = setTimeout(function() { t.className = 'toast'; }, 2200);
}
